#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;


using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
#endregion
namespace gov.va.med.vbecs.BOL.UnitTests
{
    [TestFixture]
	public class BOL_LockManager : BaseTest 
	{

		private BOL.LockManager _tstLM;
        private int _lockFormId = -1000;
        private int _baseLockFormId = -999;
        private bool _refreshData = true;

		[SetUp]
		protected void SetUp()
		{
			this.BaseSetUp();

            if (_refreshData)
            {
                this._tstLM = new LockManager(this._lockFormId, this._baseLockFormId, BOL.LockManager.DefaultLockInactivityTimeoutMin);
                this._tstLM.LockFunction = COMMON.LockFunctions.UnitTests;
                this._refreshData = false;
            }
		}

		[TearDown]
		protected void TearDown()
		{
		    this._tstLM.UnlockRecords();
		}


		[Test]
		public void Ctor_Pass()
		{
            int lockFormID = 0;
            int baseLockFormID = 0;
            BOL.LockManager tmpValue = new LockManager(lockFormID, baseLockFormID, BOL.LockManager.DefaultLockInactivityTimeoutMin);
			Assert.IsNotNull(tmpValue);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ParameterlessMethod)]
		public void Ctor_Fail()
		{
		}

		[Test]
		public void IsRecordAlreadyLockedInUC_Pass()
		{			
			System.Guid recordGuid = System.Guid.NewGuid();

            this._tstLM.LockRecord(recordGuid, Common.LockFunctions.UnitTests, false);
            BOL.Lock tmpLock2 = this._tstLM.LockRecord(recordGuid, Common.LockFunctions.UnitTests,false);
			BOL.Lock tmpLock3;

			bool retValue = this._tstLM.IsRecordAlreadyLockedInUC(recordGuid, out tmpLock3);

			Assert.AreEqual(true, retValue, "ReturnValue");
            Assert.AreEqual(tmpLock2.LockedUseCase, tmpLock3.LockedUseCase, "LockedUseCase");
		}

		[Test]
		public void IsRecordAlreadyLockedInUC_Fail()
		{
			System.Guid recordGuid = System.Guid.NewGuid();
		    BOL.Lock tmpLock1;

			bool retValue = this._tstLM.IsRecordAlreadyLockedInUC(recordGuid, out tmpLock1);

			Assert.IsNull(tmpLock1, "IsNull");
			Assert.AreEqual(false, retValue, "IsRecordAlreadyLockedInUC");
		}

		[Test]
		public void GetLock_Pass()
		{		
			System.Guid recordGuid = System.Guid.NewGuid();
			
			this._tstLM.LockRecord(recordGuid, false);
			BOL.Lock tmpValue = BOL.LockManager.GetLock(recordGuid, true);

            Assert.AreEqual(recordGuid, tmpValue.LockedRecordGuid, "LockedRecordGuid");
			Assert.AreEqual(true, tmpValue.LockExists, "LockExists");
			Assert.AreEqual(false, tmpValue.LockConflictIndicator(this._lockFormId), "LockConflictIndicator");
		}

		[Test]
		public void GetLock_Fail()
		{
			System.Guid recordGuid = System.Guid.NewGuid();

			BOL.Lock tmpValue = BOL.LockManager.GetLock(recordGuid, false);

            Assert.AreEqual(System.Guid.Empty, tmpValue.LockedRecordGuid, "LockedRecordGuid");
			Assert.AreEqual(false, tmpValue.LockExists, "LockExists");
			Assert.AreEqual(false, tmpValue.LockConflictIndicator(12345), "LockConflictIndicator");
		}


		[Test]
		public void LockRecord_Pass()
		{
			System.Guid recordGuid = System.Guid.NewGuid();

			BOL.Lock tmpValue = this._tstLM.LockRecord(recordGuid, Common.LockFunctions.UnitTests, false);

            Assert.AreEqual(recordGuid, tmpValue.LockedRecordGuid, "LockedRecordGuid");
			Assert.AreEqual(true, tmpValue.LockExists, "LockExists");

			//Proving - locking the record a second time is ok
			int prevCount = this._tstLM.LockCount;
			BOL.Lock tmpValue2 = this._tstLM.LockRecord(recordGuid, Common.LockFunctions.UnitTests,false);
			Assert.AreEqual(prevCount, this._tstLM.LockCount, "LockCount");
			Assert.AreEqual(tmpValue.LockCreatedTime, tmpValue2.LockCreatedTime, "LockCreatedTime");

			//Code coverage for half of LockRecord cannot be done, due to the impracticle setup conditions needed
		}

		[Test]
		[Ignore(StandardIgnoreReasons.Untestable)]
		public void LockRecord_Fail()
		{
			//Impossible to hit the failing conditions, they require Sql problems, and record collisions
		}

		[Test]
		public void UnlockRecord_Pass()
		{
			System.Guid[] recordGuid = {System.Guid.NewGuid()};

			BOL.Lock tmpValue = this._tstLM.LockRecord(recordGuid[0], Common.LockFunctions.UnitTests,false);
			
			//Make sure the lock exists first
			if(tmpValue.LockExists)
			{
				this._tstLM.UnlockRecords(recordGuid, this._lockFormId, Common.LockFunctions.UnitTests);
				BOL.Lock tmpLock = BOL.LockManager.GetLock(recordGuid[0], false);
                Assert.AreEqual(System.Guid.Empty, tmpLock.LockedRecordGuid, "Unlock Failed");
			}
			else
				Assert.IsTrue(false, "Lock Failed");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void UnlockRecord_Fail()
		{
		}

		
		[Test]
		public void LockCount_Pass()
		{
			//Technically, this is unreachable code - the constructor sets the _locks object to a new ArrayList
			//The NULL test condition wanted here, cannot be reached
            BOL.LockManager tmpLM = new LockManager(543210, 543210, BOL.LockManager.DefaultLockInactivityTimeoutMin);
            tmpLM.LockFunction = COMMON.LockFunctions.UnitTests;
		   
            Assert.AreEqual(0, tmpLM.LockCount, "Default");

			System.Guid recGuid = System.Guid.NewGuid();

			tmpLM.LockRecord(recGuid, false);

			Assert.AreEqual(1, tmpLM.LockCount, "After Lock");

            tmpLM.UnlockRecords();
		}

		
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void LockCount_Fail()
		{
		}


		[Test]
		public void UnlockAllRecordsForUseCaseUser_Pass()
		{
			System.Guid recordGuid = System.Guid.NewGuid();

			this._tstLM.LockRecord(recordGuid, Common.LockFunctions.UnitTests, false);
			
			this._tstLM.UnlockAllRecordsForUseCaseUser(Common.LockFunctions.UnitTests, _lockFormId, _baseLockFormId);

			BOL.Lock tmpLock = BOL.LockManager.GetLock(recordGuid, false);

            Assert.AreEqual(System.Guid.Empty, tmpLock.LockedRecordGuid, "Unlock Failed");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void UnlockAllRecordsForUseCaseUser_Fail()
		{
		}


		[Test]
		public void UnlockAllRecordsForUser_Pass()
		{
			System.Guid recordGuid = System.Guid.NewGuid();

            this._tstLM.LockRecord(recordGuid, true);
			
			BOL.LockManager.UnlockAllRecordsForUser(Common.LogonUser.LogonUserName, true);

			BOL.Lock tmpLock = BOL.LockManager.GetLock(recordGuid, false);

            Assert.AreEqual(System.Guid.Empty, tmpLock.LockedRecordGuid, "UnlockAll Failed");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void UnlockAllRecordsForUser_Fail()
		{
		}

        [Test]
        public void ExpiredBaseLockedFormIds_Pass()
        {
            ArrayList val = BOL.LockManager.ExpiredBaseLockedFormIds;

            Assert.NotNull(val, "ExpiredBaseLockedFormIds");
        }

        [Test]
        [Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
        public void ExpiredBaseLockedFormIds_Fail()
        {
        }

        [Test]
        public void AddToLockAlertList_Pass()
        {
            BOL.LockManager.AddToLockAlertList(_baseLockFormId, _lockFormId);

            Assert.False(BOL.LockManager.AllLockAlertFormsAreClosed(_baseLockFormId));

            BOL.LockManager.RemoveFromLockAlertList(_lockFormId);
        }

        [Test]
        [Ignore(StandardIgnoreReasons.NoFailureConditions)]
        public void AddToLockAlertList_Fail()
        {
        }

        [Test]
        public void RemoveFromLockAlertList_Pass()
        {
            BOL.LockManager.AddToLockAlertList(_baseLockFormId, _lockFormId);

            Assert.False(BOL.LockManager.AllLockAlertFormsAreClosed(_baseLockFormId));

            BOL.LockManager.RemoveFromLockAlertList(_lockFormId);
        }

        [Test]
        [Ignore(StandardIgnoreReasons.NoFailureConditions)]
        public void RemoveFromLockAlertList_Fail()
        {
        }

        [Test]
        public void AllLockAlertFormsAreClosed_Pass()
        {
            BOL.LockManager.AddToLockAlertList(_baseLockFormId, _lockFormId);

            Assert.False(BOL.LockManager.AllLockAlertFormsAreClosed(_baseLockFormId));

            BOL.LockManager.RemoveFromLockAlertList(_lockFormId);
        }

        [Test]
        [Ignore(StandardIgnoreReasons.NoFailureConditions)]
        public void AllLockAlertFormsAreClosed_Fail()
        {
        }

        [Test]
        public void HasBaseLockTimedout_Pass()
        {
            BOL.LockManager.ExpiredBaseLockedFormIds.Add(_baseLockFormId);

            Assert.True(BOL.LockManager.HasBaseLockTimedout(_baseLockFormId));
        }

        [Test]
        [Ignore(StandardIgnoreReasons.NoFailureConditions)]
        public void HasBaseLockTimedout_Fail()
        {
        }

        [Test]
        public void IsLockCarrier_Pass()
        {
            Assert.True(this._tstLM.IsLockCarrier(_lockFormId));
        }

        [Test]
        public void IsLockCarrier_Fail()
        {
            Assert.True(this._tstLM.IsLockCarrier(0));
        }

        [Test]
        public void LockUseCase_Pass()
        {
            this._refreshData = true;   //we'll need new stuff..

            Guid tmpGuid = Guid.NewGuid();
            BOL.Lock tstLock = this._tstLM.LockUseCase(tmpGuid, false);
            Assert.NotNull(tstLock, "Lock object exists");
            Assert.AreEqual(tmpGuid, tstLock.LockedRecordGuid, "LockedRecordGuid");

            this._tstLM.UnlockRecords();
        }

        [Test]
        [Ignore(StandardIgnoreReasons.NoFailureConditions)]
        public void LockUseCase_Fail()
        {
        }

        [Test]
        public void UnlockRecords_Pass()
        {
            this._refreshData = true;   //we'll need new stuff..

            Guid tmpGuid = Guid.NewGuid();
            BOL.Lock tstLock = this._tstLM.LockRecord(tmpGuid, false);

            Assert.AreEqual(this._tstLM.LockCount, 1, "LockCount");

            this._tstLM.UnlockRecords();

            Assert.AreEqual(this._tstLM.LockCount, 0, "LockCount-AfterUnlock");
        }

        [Test]
        [Ignore(StandardIgnoreReasons.NoFailureConditions)]
        public void UnlockRecords_Fail()
        {
        }

        [Test]
        public void RecordGuids_Pass()
        {
            this._refreshData = true;   //we'll need new stuff..

            Guid tmpGuid = Guid.NewGuid();
            BOL.Lock tstLock = this._tstLM.LockRecord(tmpGuid, false);

            Assert.AreEqual(this._tstLM.RecordGuids.Count, 1, "Count");
            
            this._tstLM.UnlockRecords();

            Assert.AreEqual(this._tstLM.RecordGuids.Count, 0, "Count-AfterUnlock");
        }

        [Test]
        [Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
        public void RecordGuids_Fail()
        {
        }

        [Test]
        public void GUIFriendlyLockLocation_Pass()
        {
            string val = BOL.LockManager.GUIFriendlyLockLocation(Common.LockFunctions.UnitTests);

            Assert.NotNull(val);
        }

        [Test]
        [Ignore(StandardIgnoreReasons.NoFailureConditions)]
        public void GUIFriendlyLockLocation_Fail()
        {
        }

        [Test]
        public void LockFunction_Pass()
        {
            Common.LockFunctions val = this._tstLM.LockFunction;

            Assert.AreEqual(val, Common.LockFunctions.UnitTests, "UnitTests");
        }

        [Test]
        [Ignore(StandardIgnoreReasons.NoFailureConditions)]
        public void LockFunction_Fail()
        {
        }


        [Test]
        public void LockingConflict_Pass()
        {
            //TODO: someday
        }

        [Test]
        [Ignore(StandardIgnoreReasons.NoFailureConditions)]
        public void LockingConflict_Fail()
        {
        }


        [Test]
        public void LockedFormId_Pass()
        {
            Assert.AreEqual(_lockFormId, this._tstLM.LockedFormId);
        }

        [Test]
        [Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
        public void LockedFormId_Fail()
        {
        }


        [Test]
        public void BaseLockedFormId_Pass()
        {
            Assert.AreEqual(_baseLockFormId, this._tstLM.BaseLockedFormId);
        }

        [Test]
        [Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
        public void BaseLockedFormId_Fail()
        {
        }


	}
}
#endif


